/*
    Copyright (C) 2020 Daniel Schultz

    This file is part of FLINT.

    FLINT is free software: you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.  See <https://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include "nmod_mpoly.h"

int
main(void)
{
    slong i, j;
    FLINT_TEST_INIT(state);

    flint_printf("scalar_addmul_ui....");
    fflush(stdout);

    for (i = 0; i < 10 * flint_test_multiplier(); i++)
    {
        nmod_mpoly_ctx_t ctx;
        nmod_mpoly_t f, g, h, k;
        mp_limb_t modulus;
        slong len, len1, len2;
        slong exp_bits, exp_bits1, exp_bits2;

        modulus = n_randint(state, FLINT_BITS - 1) + 1;
        modulus = n_randbits(state, modulus);
        modulus = FLINT_MAX(2, modulus);

        nmod_mpoly_ctx_init_rand(ctx, state, 20, modulus);

        nmod_mpoly_init(f, ctx);
        nmod_mpoly_init(g, ctx);
        nmod_mpoly_init(h, ctx);
        nmod_mpoly_init(k, ctx);

        len = n_randint(state, 100);
        len1 = n_randint(state, 100);
        len2 = n_randint(state, 100);

        exp_bits = n_randint(state, 200) + 2;
        exp_bits1 = n_randint(state, 200) + 2;
        exp_bits2 = n_randint(state, 200) + 2;

        for (j = 0; j < 10; j++)
        {
            mp_limb_t c = n_randlimb(state);
            nmod_mpoly_randtest_bits(f, state, len1, exp_bits1, ctx);
            nmod_mpoly_randtest_bits(g, state, len2, exp_bits2, ctx);
            nmod_mpoly_randtest_bits(h, state, len, exp_bits, ctx);
            nmod_mpoly_randtest_bits(k, state, len, exp_bits, ctx);

            nmod_mpoly_scalar_addmul_ui(f, g, h, c, ctx);

            nmod_mpoly_scalar_mul_ui(k, h, c, ctx);
            nmod_mpoly_add(k, k, g, ctx);

            if (!nmod_mpoly_equal(f, k, ctx))
            {
                flint_printf("FAIL: check definition\n");
                flint_printf("i = %wd, j = %wd\n", i ,j);
                flint_abort();
            }

            nmod_mpoly_set(k, g, ctx);
            nmod_mpoly_scalar_addmul_ui(k, k, h, c, ctx);
            if (!nmod_mpoly_equal(f, k, ctx))
            {
                flint_printf("FAIL: check aliasing first argument\n");
                flint_printf("i = %wd, j = %wd\n", i ,j);
                flint_abort();
            }

            nmod_mpoly_set(k, h, ctx);
            nmod_mpoly_scalar_addmul_ui(k, g, k, c, ctx);
            if (!nmod_mpoly_equal(f, k, ctx))
            {
                flint_printf("FAIL: check aliasing second argument\n");
                flint_printf("i = %wd, j = %wd\n", i ,j);
                flint_abort();
            }
        }

        nmod_mpoly_clear(f, ctx);
        nmod_mpoly_clear(g, ctx);
        nmod_mpoly_clear(h, ctx);
        nmod_mpoly_clear(k, ctx);

        nmod_mpoly_ctx_clear(ctx);
    }

    FLINT_TEST_CLEANUP(state);
    
    flint_printf("PASS\n");
    return 0;
}

