/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* SCROLL.C --- Scroll bars. */

#include <stdlib.h>
#include <assert.h>
#include <graphics.h>
#include "window.h"
#include "wintern.h"
#include "resource.h"
#include "key.h"

enum {

scrollbar_border_width = 1,
scrollbar_bg_color = cDARKGRAY,
scrollbar_fg_color = cBLACK,
scrollbar_border_color = scrollbar_fg_color,
scrollbar_shadow_width = 2,
scrollbar_button = bLEFT,
scrollbar_focus_color = cLIGHTRED,
scrollbar_y_margin = 3,

};

LOCAL(void) locate_slider(SCROLLBAR sb)
{
  int slide_max, x, y;
  BUTTON s;

  s = &sb->slider;
  if (sb_status_p(sb, sbENABLED)) {
    if (sb_status_p(sb, sbVERTICAL)) {
      x = 0;
      y = sb->arrow0.window.height + scrollbar_border_width;
      if (sb->n_pos > 1) {
	slide_max = sb->window.height - y - sb->arrow1.window.height - scrollbar_border_width - sb->slider.window.height;
	y += scale_int(slide_max, sb->pos, sb->n_pos - 1);
      }
    }
    else {
      y = 0;
      x = sb->arrow0.window.width + scrollbar_border_width;
      if (sb->n_pos > 1) {
	slide_max = sb->window.width - x - sb->arrow1.window.width - scrollbar_border_width - sb->slider.window.width;
	x += scale_int(slide_max, sb->pos, sb->n_pos - 1);
      }
    }
    (sb->pos == 0             ? disable_button : enable_button)(&sb->arrow0);
    (sb->pos == sb->n_pos - 1 ? disable_button : enable_button)(&sb->arrow1);
    locate_window(&sb->slider.window, x, y, 0);
    if (sb->n_pos == 1)
      s->status &= notbit(bENABLED);
    else
      s->status |= bit(bENABLED);
    draw_button(s);
    sb->window.event_mask |= bit(eBUTTON_PRESS)
			    |bit(eGAIN_FOCUS)
			    |bit(eVISIBLE_HOTKEY);
  }
  else {
    disable_button(&sb->arrow0);
    disable_button(&sb->arrow1);
    s->status &= notbit(bENABLED);
    draw_button(&sb->slider);
    sb->window.event_mask &= notbit(eMOUSE_MOTION)
			    &notbit(eBUTTON_PRESS)
			    &notbit(eGAIN_FOCUS)
			    &notbit(eVISIBLE_HOTKEY);
  }
}

/* Execute action accounting for position reversal in report. */
LOCAL(void) do_action(SCROLLBAR sb)
{
  (*sb->action.code)(sb_status_p(sb, sbREVERSE) ? (sb->n_pos - 1) - sb->pos : sb->pos, sb->action.env);
}

/* Move a slider a relative amount, checking
   for bounds and unnecessary updates. */
LOCAL(void) bump_slider(SCROLLBAR sb, int d)
{
  int p = sb->pos + d;
  if (p < 0)
    p = 0;
  else if (p >= sb->n_pos)
    p = sb->n_pos - 1;
  if (p != sb->pos) {
    sb->pos = p;
    locate_slider(sb);
    do_action(sb);
  }
}

#define sb ((SCROLLBAR)env)
static void move0_action(ENV env) { bump_slider(sb, -sb->n_inc_pos); }
static void move1_action(ENV env) { bump_slider(sb,  sb->n_inc_pos); }
static void step0_action(ENV env) { bump_slider(sb, -sb->n_step_pos); }
static void step1_action(ENV env) { bump_slider(sb,  sb->n_step_pos); }
#undef sb

static enum { STATIC, STEPPING_TO_0, STEPPING_TO_1, MOVING } slider_status;

LOCAL(void) handle_bg_press(EVENT e)
{
  int s0, s1;
  SCROLLBAR sb;

  if (e->mouse.button != scrollbar_button)
    return;

  sb = (SCROLLBAR)e->mouse.window;
  if (sb_status_p(sb, sbVERTICAL)) {
    s0 = window_y(&sb->slider.window);
    s1 = s0 + sb->slider.window.height - 1;
    slider_status = (e->mouse.y < s0) ? STEPPING_TO_0 :
		    (e->mouse.y > s1) ? STEPPING_TO_1 : MOVING;
  }
  else {
    s0 = window_x(&sb->slider.window);
    s1 = s0 + sb->slider.window.width - 1;
    slider_status = (e->mouse.x < s0) ? STEPPING_TO_0 :
		    (e->mouse.x > s1) ? STEPPING_TO_1 : MOVING;
  }
  switch (slider_status) {

    case STEPPING_TO_0:
      begin_repeat(step0_action, e->mouse.window);
      break;

    case STEPPING_TO_1:
      begin_repeat(step1_action, e->mouse.window);
      break;

    case MOVING:
      e->mouse.window->event_mask |= bit(eMOUSE_MOTION);
      break;
  }
}

LOCAL(void) handle_slider_motion(EVENT e)
{
  int c0, c1, x, y, ss;
  SCROLLBAR sb = (SCROLLBAR)e->mouse.window;

  if (sb_status_p(sb, sbVERTICAL)) {
    c0 = sb->arrow0.window.height + scrollbar_border_width;
    ss = sb->slider.window.height;
    c1 = sb->window.height - sb->arrow1.window.height - scrollbar_border_width - ss;
    x = 0;
    y = e->mouse.y - (ss >> 1);
    if (y < c0)
      y = c0;
    else if (y > c1)
      y = c1;
  }
  else {
    c0 = sb->arrow0.window.width + scrollbar_border_width;
    ss = sb->slider.window.width;
    c1 = sb->window.width - sb->arrow1.window.width - scrollbar_border_width - ss;
    x = e->mouse.x - (ss >> 1);
    if (x < c0)
      x = c0;
    else if (x > c1)
      x = c1;
    y = 0;
  }
  locate_window(&sb->slider.window, x, y, 0);
}

LOCAL(void) handle_bg_release(EVENT e)
{
  int old_pos;
  SCROLLBAR sb; 

  if (e->mouse.button != scrollbar_button)
    return;

  sb = (SCROLLBAR)e->mouse.window;

  switch (slider_status) {

    case STEPPING_TO_0:
    case STEPPING_TO_1:
      end_repeat();
      break;

    case MOVING:
      sb->window.event_mask &= notbit(eMOUSE_MOTION);
      old_pos = sb->pos;
      if (sb_status_p(sb, sbVERTICAL)) {
	int slide_start = sb->arrow0.window.height + scrollbar_border_width;
	int slide_max = sb->window.height - slide_start - sb->arrow1.window.height - scrollbar_border_width - sb->slider.window.height;
	sb->pos = scale_int((window_y(&sb->slider.window) - slide_start), (sb->n_pos - 1), slide_max);
      }
      else {
	int slide_start = sb->arrow0.window.width + scrollbar_border_width;
	int slide_max = sb->window.width - slide_start - sb->arrow1.window.width - scrollbar_border_width - sb->slider.window.width;
	sb->pos = scale_int((window_x(&sb->slider.window) - slide_start), (sb->n_pos - 1), slide_max);
      }
      locate_slider(sb);
      if (sb->pos != old_pos)
	do_action(sb);
      break;

  }
  slider_status = STATIC;
}

int position_scrollbar_by_key(SCROLLBAR sb, int key, unsigned mask)
{
  int d = (mask & bit(ckSCROLL_LOCK)) ? sb->n_step_pos : sb->n_inc_pos;

  if (sb_status_p(sb, sbVERTICAL))
    switch (key) {
      case '\t':	unset_focus(&sb->window);	return 1;
      case UP_ARROW:	d = -d;				break;
      case DOWN_ARROW:					break;
      default:		return 0;
    }
  else 
    switch (key) {
      case '\t':	unset_focus(&sb->window);	return 1;
      case LEFT_ARROW:	d = -d;				break;
      case RIGHT_ARROW:					break;
      default:		return 0;
    }
  bump_slider(sb, d);
  return 1;
}

LOCAL(void) handle_map(EVENT e)
{
  SCROLLBAR sb = (SCROLLBAR)e->map.window;

  locate_slider(sb);
  if (sb->title.str != NULL) {
    push_graphics_state(&sb->window, 0);
    protect_cursor(&sb->window);
    setcolor(scrollbar_fg_color);
    settextjustify(LEFT_TEXT, BOTTOM_TEXT);
    out_hot_textxy(0, -scrollbar_y_margin, &sb->title);
    unprotect_cursor();
    pop_graphics_state();
  }
}

LOCAL(void) handle_hotkey(EVENT e)
{
  SCROLLBAR sb = (SCROLLBAR)e->hotkey.window;
  if (key_hot_p(e->hotkey.key, &sb->title))
    set_focus(&sb->window);
}

BeginDefDispatch(scrollbar)
  Dispatch(eMAP, handle_map)
  Dispatch(eBUTTON_PRESS, handle_bg_press)
  Dispatch(eBUTTON_RELEASE, handle_bg_release)
  Dispatch(eMOUSE_MOTION, handle_slider_motion)
  DispatchAction(eGAIN_FOCUS, set_window_border_color(e->focus.window, scrollbar_focus_color))
  DispatchAction(eLOSE_FOCUS, set_window_border_color(e->focus.window, scrollbar_border_color))
  DispatchAction(eKEYSTROKE, if (!position_scrollbar_by_key((SCROLLBAR)e->keystroke.window, e->keystroke.key, e->keystroke.mask)) refuse_keystroke(e))
  Dispatch(eVISIBLE_HOTKEY, handle_hotkey)
EndDefDispatch(scrollbar)

int scrollbar_thickness(int n_borders)
{
  assert(uarrow_width == larrow_width);
  return(uarrow_width + 2 * scrollbar_shadow_width + n_borders * scrollbar_border_width);
}

void position_scrollbar(SCROLLBAR sb, int pos)
{
  if (pos >= sb->n_pos) 
    pos = sb->n_pos - 1;
  else if (pos < 0) 
    pos = 0;
  sb->pos = sb_status_p(sb, sbREVERSE) ? (sb->n_pos - 1) - pos : pos;
  locate_slider(sb);
}

/* Make a new scrollbar. */
void open_scrollbar(SCROLLBAR sb, int x, int y, WINDOW parent)
{
  int width, height, x_a1, y_a1;
  BUTTON s = &sb->slider;

  if (sb_status_p(sb, sbVERTICAL)) {
    width = scrollbar_thickness(0);
    height = sb->length == FULL_LENGTH ? 
	       sb_status_p(sb, sb1OF2) ?
		 parent->height - width - scrollbar_border_width :
		 parent->height :
	       sb->length;
    if (x == DEFAULT)
      x = parent->width - width;
    if (y == DEFAULT)
      y = 0;
    x_a1 = 0;
    y_a1 = height - width;
  }
  else {
    height = scrollbar_thickness(0);
    width = sb->length == FULL_LENGTH ? 
	      sb_status_p(sb, sb1OF2) ?
		parent->width - height - scrollbar_border_width :
		parent->width :
	      sb->length;
    if (x == DEFAULT)
      x = 0;
    if (y == DEFAULT)
      y = parent->height - height;
    x_a1 = width - height;
    y_a1 = 0;
  }

  if (sb_status_p(sb, sbREVERSE)) 
    sb->pos = (sb->n_pos - 1) - sb->pos;

  open_window(&sb->window, parent, x, y, width, height,
	      scrollbar_border_width, scrollbar_border_color, scrollbar_bg_color,
	      bit(eMAP)|bit(eBUTTON_RELEASE)|bit(eLOSE_FOCUS)|bit(eKEYSTROKE));
  SetDispatch(&sb->window, scrollbar);

  if(sb_status_p(sb, sbVERTICAL)) {
    SetBitmapButtonDesc(&sb->arrow0, uarrow, 0, scrollbar_shadow_width, scrollbar_border_width, Repeat, move0_action, sb);
    SetBitmapButtonDesc(&sb->arrow1, darrow, 0, scrollbar_shadow_width, scrollbar_border_width, Repeat, move1_action, sb);
  }                          
  else {
    SetBitmapButtonDesc(&sb->arrow0, larrow, 0, scrollbar_shadow_width, scrollbar_border_width, Repeat, move0_action, sb);
    SetBitmapButtonDesc(&sb->arrow1, rarrow, 0, scrollbar_shadow_width, scrollbar_border_width, Repeat, move1_action, sb);
  }
  open_button(&sb->arrow0, 0, 0, &sb->window);
  map_window(&sb->arrow0.window);
  open_button(&sb->arrow1, x_a1, y_a1, &sb->window);
  map_window(&sb->arrow1.window);

  /* Build up slider as a special kind of button. */
  SetBitmapButtonDesc(s, slider, NoHotKey, scrollbar_shadow_width, scrollbar_border_width, NoFlags, null_action_code, NULL);
  open_window(&s->window, &sb->window, 0, 0,
	      button_width(s), button_height(s),
	      0, NO_COLOR, NO_COLOR, bit(eMAP));
  s->window.handler = handle_button_map;
  map_window(&s->window);
}

/* Make a scrollbar ready for mouse touches. */
void enable_scrollbar(SCROLLBAR sb)
{
  sb->status |= bit(sbENABLED);
  locate_slider(sb);
}

/* Turn off a scrollbar. */
void disable_scrollbar(SCROLLBAR sb)
{
  sb->status &= notbit(sbENABLED);
  locate_slider(sb);
  unset_focus(&sb->window);
}
